# Web Console - Guía de Instalación para Linux (Entorno Productivo)

Esta guía te ayudará a configurar y desplegar el Web Console en un entorno productivo Linux.

## Introducción

El Web Console es una aplicación web desarrollada en Go con frontend React que permite conectarse a servidores Linux mediante SSH directamente desde el navegador. Esta aplicación se integra con Cluster Panel para proporcionar una interfaz web para la gestión remota de servidores.

## Requisitos Previos

Antes de comenzar, asegúrate de tener instalado lo siguiente en tu servidor Linux:

- **Go 1.13 o superior**: [Instalación de Go](https://golang.org/doc/install)
  - Verificar instalación: `go version`
- **Node.js 18**: [Instalación de Node.js](https://nodejs.org/)
  - Verificar instalación: `node --version`
- **Yarn**: Instalar con npm o desde el repositorio
  ```bash
  npm install -g yarn
  # o
  curl -sS https://dl.yarnpkg.com/debian/pubkey.gpg | sudo apt-key add -
  echo "deb https://dl.yarnpkg.com/debian/ stable main" | sudo tee /etc/apt/sources.list.d/yarn.list
  sudo apt update && sudo apt install yarn
  ```
  - Verificar instalación: `yarn --version`
- **Statik**: Herramienta para embeber archivos estáticos en el binario Go
  - Instalación: `go install github.com/rakyll/statik@latest`
  - Nota: Asegúrate de que `$HOME/go/bin` esté en tu PATH
- **Sistema de Servicios**: systemd (incluido por defecto en la mayoría de distribuciones Linux modernas)
- **Git**: Para clonar el repositorio (si es necesario)
- **Firewall**: ufw o firewalld (dependiendo de tu distribución)

## Instalación de Dependencias

### 1. Actualizar el Sistema

```bash
sudo apt update && sudo apt upgrade -y
# o para RHEL/CentOS:
# sudo yum update -y
```

### 2. Instalar Go

```bash
# Descargar Go (ajusta la versión según la última disponible)
wget https://go.dev/dl/go1.21.5.linux-amd64.tar.gz
sudo rm -rf /usr/local/go
sudo tar -C /usr/local -xzf go1.21.5.linux-amd64.tar.gz

# Agregar Go al PATH
echo 'export PATH=$PATH:/usr/local/go/bin:$HOME/go/bin' >> ~/.bashrc
source ~/.bashrc

# Verificar instalación
go version
```

### 3. Instalar Node.js y Yarn

**Para Ubuntu/Debian:**

```bash
curl -fsSL https://deb.nodesource.com/setup_18.x | sudo -E bash -
sudo apt install -y nodejs
sudo npm install -g yarn
```

**Para RHEL/CentOS:**

```bash
curl -fsSL https://rpm.nodesource.com/setup_18.x | sudo bash -
sudo yum install -y nodejs
sudo npm install -g yarn
```

### 4. Instalar Statik

```bash
go install github.com/rakyll/statik@latest
# Verificar que está en el PATH
statik --help
```

### 5. Clonar o Preparar el Proyecto

Si aún no tienes el código fuente:

```bash
cd /opt  # o el directorio de tu elección
sudo mkdir -p /opt/web-console
sudo chown $USER:$USER /opt/web-console
cd /opt/web-console
# Clonar desde tu repositorio o copiar los archivos
```

O si ya tienes los archivos, navega al directorio:

```bash
cd /ruta/a/web-console
```

## Configuración

### 1. Instalar Dependencias de Go

```bash
go mod download
go mod verify
```

### 2. Instalar Dependencias del Frontend React

```bash
cd web
yarn install
cd ..
```

### 3. Configurar el Archivo de Configuración

El archivo de configuración principal se encuentra en `conf/config.yaml`. Este archivo controla la configuración del servidor.

Crea o edita el archivo `conf/config.yaml`:

```bash
mkdir -p conf
nano conf/config.yaml
```

Configuración de ejemplo para producción:

```yaml
site:
  appname: ssh-web-console
  listen_addr: :2222
  runmode: prod
  deploy_host: tu-dominio.com # Cambia por tu dominio real

prod:
  # http path of static files and views
  static_prefix: /

dev: # config used in debug mode.
  # https prefix of static files only
  static_prefix: /static/
  static_redirect: "localhost:8080"
  static_dir: ./dist/
  views_prefix: /
  views_dir: views/

ssh:
  buffer_checker_cycle_time: 60

jwt:
  jwt_secret: tu-secret-jwt-muy-seguro-cambiar-en-produccion # ¡CAMBIAR EN PRODUCCIÓN!
  token_lifetime: 7200
  issuer: tu-dominio.com
  query_token_key: _t
```

**Importante para producción:**

- `listen_addr: :2222` - Puerto en el que escuchará el servidor
- `runmode: prod` - Debe estar en modo producción
- `deploy_host`: Configura tu dominio real
- `jwt_secret`: **Cambia esto por un secreto fuerte y único en producción**

### 4. Configurar Variables de Entorno React

Crea un archivo `.env` en el directorio `web/` con las variables de entorno necesarias:

```bash
cd web
nano .env
```

Contenido del archivo `.env`:

```env
REACT_APP_CLUSTER_URL=https://tu-cluster-panel.com
REACT_APP_API_URL=tu-dominio.com:2222
REACT_APP_ROUTER_BASE=
REACT_APP_API_HTTPS=true
```

**Nota:**

- `REACT_APP_CLUSTER_URL`: URL base completa de tu instalación de Cluster Panel (sin barra final)
- `REACT_APP_API_URL`: URL del servidor web console (opcional)
- `REACT_APP_API_HTTPS`: Configurar como `true` si usas HTTPS

```bash
cd ..
```

## Compilación

### Paso 1: Compilar el Frontend React

```bash
cd web
yarn build
cd ..
```

Esto generará los archivos estáticos en `web/build/`.

### Paso 2: Generar Archivos Statik

Statik embebe los archivos estáticos del frontend en el código Go:

```bash
statik --src=web/build
```

Esto generará archivos en el directorio `statik/`.

### Paso 3: Compilar el Servidor Go

**Compilación estándar:**

```bash
go build -o ssh-web-console-linux-amd64
```

**Compilación optimizada para producción:**

```bash
CGO_ENABLED=0 GOOS=linux GOARCH=amd64 go build -o ssh-web-console-linux-amd64
```

Esto genera un binario estático que no depende de librerías del sistema.

**Para arquitectura ARM64:**

```bash
CGO_ENABLED=0 GOOS=linux GOARCH=arm64 go build -o ssh-web-console-linux-arm64
```

### Paso 4: Verificar el Binario

```bash
file ssh-web-console-linux-amd64
ls -lh ssh-web-console-linux-amd64
```

## Despliegue

### 1. Crear Directorio de Despliegue

```bash
sudo mkdir -p /opt/web-console
sudo mkdir -p /opt/web-console/conf
sudo mkdir -p /opt/web-console/logs
```

### 2. Copiar Archivos

```bash
# Copiar el binario
sudo cp ssh-web-console-linux-amd64 /opt/web-console/ssh-web-console
sudo chmod +x /opt/web-console/ssh-web-console

# Copiar el archivo de configuración
sudo cp conf/config.yaml /opt/web-console/conf/config.yaml

# Ajustar permisos
sudo chown -R webconsole:webconsole /opt/web-console
```

**Nota:** Si el usuario `webconsole` no existe, créalo:

```bash
sudo useradd -r -s /bin/false webconsole
```

### 3. Configurar como Servicio systemd

Crea un archivo de servicio systemd:

```bash
sudo nano /etc/systemd/system/web-console.service
```

Contenido del archivo de servicio:

```ini
[Unit]
Description=Web Console SSH Terminal Server
After=network.target

[Service]
Type=simple
User=webconsole
Group=webconsole
WorkingDirectory=/opt/web-console
ExecStart=/opt/web-console/ssh-web-console -config /opt/web-console/conf/config.yaml
Restart=always
RestartSec=10
StandardOutput=journal
StandardError=journal
SyslogIdentifier=web-console

# Seguridad
NoNewPrivileges=true
PrivateTmp=true
ProtectSystem=strict
ProtectHome=true
ReadWritePaths=/opt/web-console/logs
ReadOnlyPaths=/opt/web-console/conf

[Install]
WantedBy=multi-user.target
```

**Habilitar y Iniciar el Servicio:**

```bash
# Recargar systemd
sudo systemctl daemon-reload

# Habilitar el servicio para iniciar al arrancar
sudo systemctl enable web-console

# Iniciar el servicio
sudo systemctl start web-console

# Verificar el estado
sudo systemctl status web-console
```

### 4. Verificar Logs

```bash
# Ver logs en tiempo real
sudo journalctl -u web-console -f

# Ver últimos logs
sudo journalctl -u web-console -n 50
```

### 5. Configurar Firewall

**Para Ubuntu/Debian (ufw):**

```bash
sudo ufw allow 2222/tcp
sudo ufw reload
```

**Para RHEL/CentOS (firewalld):**

```bash
sudo firewall-cmd --permanent --add-port=2222/tcp
sudo firewall-cmd --reload
```

### 6. Configurar Proxy Reverso (Recomendado)

Es recomendable usar un proxy reverso como Nginx o Apache para:

- Terminación SSL/TLS (HTTPS)
- Enrutamiento
- Balanceo de carga (si es necesario)

**Ejemplo de configuración Nginx:**

```nginx
upstream web-console {
    server 127.0.0.1:2222;
    keepalive 64;
}

server {
    listen 80;
    listen [::]:80;
    server_name console.tu-dominio.com;

    # Redirigir HTTP a HTTPS
    return 301 https://$server_name$request_uri;
}

server {
    listen 443 ssl http2;
    listen [::]:443 ssl http2;
    server_name console.tu-dominio.com;

    ssl_certificate /etc/letsencrypt/live/tu-dominio.com/fullchain.pem;
    ssl_certificate_key /etc/letsencrypt/live/tu-dominio.com/privkey.pem;
    ssl_protocols TLSv1.2 TLSv1.3;
    ssl_ciphers HIGH:!aNULL:!MD5;

    location / {
        proxy_pass http://web-console;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection "upgrade";
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
        proxy_read_timeout 86400;
        proxy_buffering off;
    }
}
```

Después de configurar Nginx:

```bash
sudo nginx -t  # Verificar configuración
sudo systemctl reload nginx  # Recargar Nginx
```

## Verificación

### 1. Verificar que el Servicio está Ejecutándose

```bash
sudo systemctl status web-console
```

Deberías ver algo como:

```
Active: active (running) since ...
```

### 2. Verificar que el Puerto está Escuchando

```bash
sudo netstat -tlnp | grep 2222
# o
sudo ss -tlnp | grep 2222
```

Deberías ver que el puerto 2222 está en estado LISTEN.

### 3. Acceder desde el Navegador

Abre tu navegador y visita:

- Directo al servidor: `http://tu-servidor:2222`
- A través del proxy: `https://console.tu-dominio.com`

Deberías ver la interfaz de login del Web Console.

### 4. Probar Conexión SSH

Ingresa:

- Host: IP o hostname del servidor al que quieres conectarte
- Puerto: 22 (por defecto)
- Usuario: Tu usuario SSH
- Contraseña: Tu contraseña SSH

## Integración con Cluster Panel

Para que el Web Console funcione integrado con Cluster Panel:

1. **Configurar la URL en Cluster Panel:**

   En la configuración de Cluster Panel, asegúrate de que `urlWebConsole` esté configurada con la URL del Web Console:

   - Si usas proxy reverso: `https://console.tu-dominio.com`
   - Si accedes directamente: `http://tu-servidor:2222`

2. **Variable de Entorno REACT_APP_CLUSTER_URL:**

   Al compilar el frontend, la variable `REACT_APP_CLUSTER_URL` debe apuntar a la URL de tu Cluster Panel:

   ```env
   REACT_APP_CLUSTER_URL=https://tu-cluster-panel.com
   ```

   **Importante:** Después de cambiar esta variable, debes recompilar el frontend:

   ```bash
   cd web
   yarn build
   cd ..
   statik --src=web/build
   go build -o ssh-web-console-linux-amd64
   sudo systemctl restart web-console
   ```

3. **Probar la Integración:**

   Desde Cluster Panel, al acceder a la opción de Web Console, debería cargarse el Web Console en un iframe con las credenciales ya configuradas.

## Mantenimiento

### Reiniciar el Servicio

```bash
sudo systemctl restart web-console
```

### Detener el Servicio

```bash
sudo systemctl stop web-console
```

### Iniciar el Servicio

```bash
sudo systemctl start web-console
```

### Ver Logs en Tiempo Real

```bash
sudo journalctl -u web-console -f
```

### Actualizar el Binario

```bash
# Compilar nueva versión (ver sección de Compilación)
# Detener el servicio
sudo systemctl stop web-console

# Hacer backup del binario actual
sudo cp /opt/web-console/ssh-web-console /opt/web-console/ssh-web-console.backup

# Copiar nuevo binario
sudo cp ssh-web-console-linux-amd64 /opt/web-console/ssh-web-console
sudo chmod +x /opt/web-console/ssh-web-console

# Iniciar el servicio
sudo systemctl start web-console

# Verificar
sudo systemctl status web-console
```

## Solución de Problemas Comunes

### El servicio no inicia

**Verificar logs:**

```bash
sudo journalctl -u web-console -n 100 --no-pager
```

**Problemas comunes:**

- Archivo de configuración no encontrado: Verifica la ruta en el servicio systemd
- Permisos: Asegúrate de que el usuario `webconsole` tenga permisos sobre los archivos
- Puerto en uso: Verifica si otro proceso está usando el puerto 2222

### Error: "statik: command not found"

**Solución:**

1. Asegúrate de que Go está instalado: `go version`
2. Instala Statik: `go install github.com/rakyll/statik@latest`
3. Verifica que `$HOME/go/bin` está en tu PATH

### Error al compilar el frontend

**Solución:**

1. Verifica que Node.js y Yarn están instalados: `node --version && yarn --version`
2. Elimina `node_modules` y reinstala: `rm -rf node_modules && yarn install`
3. Verifica que las variables de entorno en `.env` están configuradas correctamente

### El Web Console no se conecta con Cluster Panel

**Solución:**

1. Verifica que `REACT_APP_CLUSTER_URL` está configurada correctamente en `web/.env`
2. Recompila el frontend después de cambiar variables de entorno
3. Verifica que la URL de Cluster Panel en `urlWebConsole` coincide exactamente
4. Verifica problemas de CORS en los logs del navegador
5. Asegúrate de que ambos servicios están ejecutándose

### Problemas de CORS o postMessage

**Solución:**

1. Asegúrate de que `REACT_APP_CLUSTER_URL` apunta exactamente a la URL de Cluster Panel (mismo protocolo, dominio y puerto)
2. Si usas HTTPS, ambos servicios deben usar HTTPS
3. Verifica que el proxy reverso (si usas uno) no esté bloqueando los headers necesarios
4. Revisa la consola del navegador para ver errores específicos

### El servicio se reinicia constantemente

**Verificar logs:**

```bash
sudo journalctl -u web-console --since "10 minutes ago" --no-pager
```

**Posibles causas:**

- Error en el archivo de configuración
- Problemas de permisos
- El puerto está en uso por otro proceso
- Errores en el binario

### Problemas de rendimiento

**Optimizaciones:**

1. Ajusta `buffer_checker_cycle_time` en `config.yaml` según tus necesidades
2. Considera usar un proxy reverso con balanceo de carga si tienes muchos usuarios
3. Monitorea los recursos del sistema: `htop` o `top`

## Seguridad en Producción

### 1. Cambiar el JWT Secret

**Importante:** Cambia el `jwt_secret` en `conf/config.yaml` por un valor único y seguro:

```bash
# Generar un secreto seguro
openssl rand -base64 32
```

Actualiza el archivo `conf/config.yaml` con el nuevo secreto.

### 2. Configurar SSL/TLS

Usa un proxy reverso (Nginx/Apache) con certificados SSL válidos (Let's Encrypt) para encriptar las conexiones.

### 3. Restringir Acceso por IP (Opcional)

Si es necesario, puedes usar firewall o el proxy reverso para restringir acceso por IP:

```bash
# Ejemplo con ufw
sudo ufw allow from IP_DESDE_LA_CUAL_PERMITIR to any port 2222
```

### 4. Actualizaciones Regulares

Mantén el sistema actualizado:

```bash
sudo apt update && sudo apt upgrade -y
```

## Usando Docker (Opcional)

Si prefieres usar Docker para despliegue:

### 1. Construir la Imagen

```bash
docker build --build-arg GOMODULE=on -t web-console:latest .
```

### 2. Ejecutar el Contenedor

```bash
docker run -d \
  --name web-console \
  -p 2222:2222 \
  -v $(pwd)/conf:/home/web/conf \
  --restart unless-stopped \
  web-console:latest
```

**Nota:** Asegúrate de que el archivo `conf/config.yaml` existe en el directorio montado.

### 3. Verificar

```bash
docker logs web-console
docker ps | grep web-console
```

## Referencias Adicionales

- **Repositorio del Proyecto**: [ssh-web-console](https://github.com/genshen/ssh-web-console)
- **Documentación de Go**: [golang.org/doc](https://golang.org/doc/)
- **Documentación de React**: [react.dev](https://react.dev/)
- **Documentación de systemd**: [systemd.io](https://systemd.io/)
- **Documentación de Nginx**: [nginx.org](https://nginx.org/en/docs/)

## Comandos Rápidos de Referencia

```bash
# Compilar frontend
cd web && yarn build && cd ..

# Generar statik
statik --src=web/build

# Compilar servidor
CGO_ENABLED=0 GOOS=linux GOARCH=amd64 go build -o ssh-web-console-linux-amd64

# Reiniciar servicio
sudo systemctl restart web-console

# Ver logs
sudo journalctl -u web-console -f

# Ver estado
sudo systemctl status web-console
```

## Notas Finales

- En producción, siempre usa HTTPS para las conexiones
- Mantén el sistema y las dependencias actualizadas
- Realiza backups regulares de la configuración
- Monitorea los logs regularmente para detectar problemas
- Considera usar un sistema de monitoreo (Prometheus, Grafana, etc.) para producción
- El Web Console usa JWT para autenticación de sesiones SSH
- Las credenciales SSH no se almacenan en el servidor, se envían directamente a través de WebSocket
